//---------------------------------------------------------------------------
#pragma hdrstop

#include <ImageBase.h>
#include <vcl.h>
#include <vcl\Registry.hpp>
#include "SigmaPiHost\SigmaPiHost.h"

USEDEF("Interface8bf.def");
USERES("Interface8bf.res");
//---------------------------------------------------------------------------
const cmdMakeMenuItem = 1; // This command can only be issued from RegisterPlugin
typedef struct
{
  pchar Menu;
  pchar Caption;
  pchar Hint;
  int Tag;
} TMakeMenuItem;

// Open new image window
const cmdCreateImageWindow = 2;
typedef struct
{
  pchar Name;
  TImageContainer Image;
} TCreateImageWindow;

// Get image from image window
const cmdGetImageWindow = 3;
typedef struct
{
  int WindowNumber;

  // Returned by Analyzer:
  int TotalWindowCount;
  pchar Name;
  TImageContainer Image;
  RECT Selection;
} TGetImageWindow;

// Get handle of main window
const cmdGetMainWindowHandle = 4;
typedef struct
{
  HWND Handle;
} TGetMainWindowHandle;

// Update progress information and refresh screen
const cmdUpdateProgress = 5;
typedef struct
{
  int Progress; // Set progress in %
} TUpdateProgress;

// Get program version number
const cmdGetProgramVersion = 6;
typedef struct 
{
  int Version; 
} TGetProgramVersion;

// Refresh children
const cmdRefreshChildren = 9;

// Function for sending cmdXXX commands to Image Analyzer
// TAnalyzerCallback function should return 0 on failure
typedef int (__stdcall *TAnalyzerCallback)(int Command, void *Data);

TAnalyzerCallback AnalyzerCallback = NULL;

TGetImageWindow FindImageWindow(const TImageContainer Image)
{
  TGetImageWindow Result;
  Result.WindowNumber = 0;
  while(AnalyzerCallback(cmdGetImageWindow,&Result))
  {
    if(Result.Image.Map==Image.Map)
      return Result;
    Result.WindowNumber++;
  }
  setmem(&Result,sizeof(Result),0);
  return Result;
}

TStringList *PluginList = NULL;

AnsiString PluginPath;

//---------------------------------------------------------------------------

int WINAPI DllEntryPoint(HINSTANCE hinst, unsigned long reason, void*)
{
  switch(reason)
  {
    case DLL_PROCESS_DETACH:
      if(PluginList)
      {
        // destroy objects
        sph_HostDestroyObjects();
        // unload SigmaPiHost.DLL
        sph_HostUnLoad();
        // delete objects
        delete PluginList;
      }
      break;
  }
  return 1;
}

//---------------------------------------------------------------------------
void ProgressCallbackFunc(int done, int total)
{
  if(total>0) done = done*100/total;
  AnalyzerCallback(cmdUpdateProgress,&done);
}

void PluginEnumFunc(char *filter_string)
{
  PluginList->Add((AnsiString)filter_string);
}

// RegisterPlugin function should return 0 on failure
extern "C" int _export RegisterPlugin(TAnalyzerCallback CallBack)
{
  int Result;
  TMakeMenuItem Item;

  AnalyzerCallback = CallBack;
  Result = AnalyzerCallback!=NULL;

  if(Result)
  {
    PluginList = new TStringList();
    try {
      // load SigmaPiHost.DLL
      sph_HostLoad(ExtractFilePath(ParamStr(0))+"ProcessingPlugins\\8bf interface\\");
      // create SigmaPiHost objects
      sph_HostCreateObjects();
    }
    catch(...)
    {
      Result = 0;
    }
  }

  if(Result)
  {
    TRegistry *Reg = new TRegistry;
    try
    {
      if(Reg->OpenKeyReadOnly("Software\\MeeSoft\\ImageAnalyzer") && Reg->ValueExists("8bf path"))
        PluginPath = Reg->ReadString("8bf path");
      else
        PluginPath = ExtractFilePath(ParamStr(0))+"ProcessingPlugins\\8bf interface\\";
    }
    __finally
    {
      delete Reg;
    }

    AnsiString CombinedPath = PluginPath;
    while(!CombinedPath.IsEmpty())
    {
      int p = CombinedPath.Pos('|');
      if(p==0) p = CombinedPath.Length()+1;
      AnsiString Path = CombinedPath.SubString(1,p-1).Trim();
      CombinedPath.Delete(1,p);
      sph_PlugInEnumerateCBF(Path.c_str(), true, (ENUMCALLBACK)PluginEnumFunc);
    }

    if(PluginList->Count>0)
    {
      for(int i=0; i<PluginList->Count; i++)
      {
        AnsiString Str = PluginList->Strings[i];
        int p = Str.Pos('\n');
        AnsiString SubMenu = "Plugins|8bf interface|"+Str.SubString(1,p-1);
        Str.Delete(1,p);
        p = Str.Pos('\n');
        AnsiString Caption = Str.SubString(1,p-1);
        Str.Delete(1,p);
        p = Str.Pos('\n');
        AnsiString Hint = "Launch "+Str.SubString(p+1,MAX_PATH);
        PluginList->Strings[i] = Str;

        Item.Tag = 2+i;
        Item.Menu = SubMenu.c_str();
        Item.Caption = Caption.c_str();
        Item.Hint = Hint.c_str();
        Result = AnalyzerCallback(cmdMakeMenuItem,&Item);
      }
      Item.Tag = 0;
      Item.Menu = "Plugins|8bf interface";
      Item.Caption = "-";
      Item.Hint = "";
      AnalyzerCallback(cmdMakeMenuItem,&Item); // Insert menu separator
    }

    Item.Tag = 1;
    Item.Menu = "Plugins|8bf interface";
    Item.Caption = "8bf interface setup...";
    Item.Hint = "|Set plugin path. Multiple paths can be separated by |.";
    AnalyzerCallback(cmdMakeMenuItem,&Item);
  }

  return Result;
}


HPALETTE MakeHPalette(pbyte Pal)
{
  PLogPalette LogPal = PLogPalette(new byte[sizeof(TLogPalette)+sizeof(TPaletteEntry)*255]);
  try
  {
    LogPal->palVersion = 0x300;
    LogPal->palNumEntries = 256;
    PPaletteEntry Entry = LogPal->palPalEntry;
    for(int i=0; i<256; i++)
    {
      Entry->peBlue = *Pal; Pal++;
      Entry->peGreen = *Pal; Pal++;
      Entry->peRed = *Pal; Pal++;
      Entry->peFlags = PC_RESERVED;
      Entry++;
    }
    return CreatePalette(LogPal);
  }
  __finally
  {
    delete LogPal;
  }
  return 0;
}

// See return codes in ImageBase header
extern "C" int __export ProcessImage(int Tag, TImageContainer *Image)
{
  if(Tag==1)
  {
    if(InputQuery("8bf plugins", "Change path for 8bf plugin files:", PluginPath))
    {
      TRegistry *Reg = new TRegistry;
      try
      {
        Reg->OpenKey("Software\\MeeSoft\\ImageAnalyzer",true);
        Reg->WriteString("8bf path",PluginPath);
      }
      __finally
      {
        delete Reg;
      }
      MessageDlg("Program must be restarted for changes to take effect", mtInformation, TMsgDlgButtons()<< mbOK, 0);
    }
  }
  else // Run 8bf plugin
  {
    AnsiString EntryPoint = PluginList->Strings[Tag-2];
    AnsiString FilterName = EntryPoint;
    int p = FilterName.Pos('\n');
    FilterName.Delete(1,p);
    EntryPoint.SetLength(p-1);

    if(sph_PlugInLoad(FilterName.c_str(), EntryPoint.c_str()))
    {
      HostRecord hRecord;
      AnalyzerCallback(cmdGetMainWindowHandle,&hRecord.hWnd);

      Graphics::TPixelFormat SourceFormat;
      switch(Image->PixelFormat)
      {
        case 0:
          sph_PlugInAbout(&hRecord); // No image, just show about box
          return 1;
        case ::pf8bit:
          SourceFormat = Graphics::pf8bit;
          break;
        case ::pf24bit:
          SourceFormat = Graphics::pf24bit;
          break;
        default:
          return 4;
      }

      TGetImageWindow Child = FindImageWindow(*Image);

      hRecord.input_mask_bitmap = NULL;
      hRecord.src_bitmap = new Graphics::TBitmap();
      try
      {
        // Setup input picture
        hRecord.src_bitmap->PixelFormat = SourceFormat;
        hRecord.src_bitmap->Width = Image->Width;
        hRecord.src_bitmap->Height = Image->Height;
        if(SourceFormat==Graphics::pf8bit)
          hRecord.src_bitmap->Palette = MakeHPalette(Image->Palette);
        for(int y=0; y<Image->Height; y++)
          memcpy(hRecord.src_bitmap->ScanLine[y], Image->Map+y*Image->BytesPerLine, Image->BytesPerLine);
        hRecord.has_bounding_rectangle = false;
        if(Child.Selection.left==-2 && Child.Image.Options) // Mask selection
        {
          hRecord.input_mask_bitmap = new Graphics::TBitmap();
          hRecord.input_mask_bitmap->PixelFormat = Graphics::pf8bit;
          hRecord.input_mask_bitmap->Width = Image->Width;
          hRecord.input_mask_bitmap->Height = Image->Height;
          for(int y=0; y<Image->Height; y++)
            memcpy(hRecord.input_mask_bitmap->ScanLine[y], Child.Image.Options+y*Image->Width, Image->Width);
        }
        else if(Child.Selection.left>=0) // Rectangle selection
        {
          hRecord.has_bounding_rectangle = true;
          hRecord.filter_rect = Child.Selection;
        }
        hRecord.output_mask_bitmap = hRecord.input_mask_bitmap;
        hRecord.output_masking_option = 0;
        hRecord.foreground_color = clBlack;         
        hRecord.background_color = clFuchsia;
        hRecord.progress_proc = (PROGRESSCALLBACK)ProgressCallbackFunc;
        int ret_code = sph_PlugInRunBMP(&hRecord);
        if(ret_code==rfCrashed)
          MessageDlg("Error occured while processing plugin request!\n"
                     "Possible reasons:\n\n"
                     "   1) Illegal plugin operation\n"
                     "   2) Operation not supported by SigmaPiHost suite.\n\n"
                     "If this is an 8 bit (palette/grayscale) picture, try changing to it 24 bit.",
                     mtError, TMsgDlgButtons()<< mbOK, 0);
        else if(ret_code==rfNoErr) 
        {
          if(hRecord.src_bitmap->PixelFormat == SourceFormat &&
             hRecord.src_bitmap->Width == Image->Width &&
             hRecord.src_bitmap->Height == Image->Height)
          {
            for(int y=0; y<Image->Height; y++)
              memcpy(Image->Map+y*Image->BytesPerLine, hRecord.src_bitmap->ScanLine[y], Image->BytesPerLine);
            return 0;
          }
          else
          {
            hRecord.src_bitmap->PixelFormat = Graphics::pf24bit;
            TCreateImageWindow NewChild;
            setmem(&NewChild, sizeof(NewChild), 0);
            AnsiString NewName = Child.Name;
            p = NewName.LastDelimiter(".\/");
            if(p>0 && NewName[p]=='.')
              NewName.SetLength(p-1);
            NewName = NewName+" output";
            NewChild.Name = NewName.c_str();
            NewChild.Image.Width = hRecord.src_bitmap->Width;
            NewChild.Image.BytesPerLine = hRecord.src_bitmap->Width*3;
            NewChild.Image.Height = hRecord.src_bitmap->Height;
            NewChild.Image.PixelFormat = ::pf24bit;
            NewChild.Image.Map = new unsigned char[NewChild.Image.BytesPerLine*NewChild.Image.Height];
            try
            {
              for(int y=0; y<NewChild.Image.Height; y++)
                memcpy(NewChild.Image.Map+y*NewChild.Image.BytesPerLine, hRecord.src_bitmap->ScanLine[y], NewChild.Image.BytesPerLine);
              AnalyzerCallback(cmdCreateImageWindow,&NewChild);
            }
            __finally
            {
              delete NewChild.Image.Map;
              if(hRecord.input_mask_bitmap)
                delete hRecord.input_mask_bitmap;
            }
          }
        }
      }
      __finally
      {
        delete hRecord.src_bitmap;
      }
    }
    else MessageDlg("Unable to load plugin.", mtError, TMsgDlgButtons()<< mbOK, 0);
  }
  return 1;
}

